// webpackの基本的な使い方 https://www.webdesignleaves.com/pr/jquery/webpack_basic_01.html

const PATH = require("path");
const terserPlugin = require("terser-webpack-plugin");


//モードの選択：'production' or 'development'をpackage.jsonのタスクの引数に記載する
const mode = process.argv.slice(2).includes("development")
  ? "development"
  : "production";


/**
 * 開発モードの時source-mapタイプのソースマップを出力
 * @see：https://webpack.js.org/configuration/devtool/
 */
const devtool = mode == "development" ? "cheap-module-source-map" : false;


// 監視モード
const watch = mode == "development" ? true : false;


// ディレクトリの設定
const dir = {
  public: PATH.join(__dirname, "..", "assets"),
  src: PATH.join(__dirname)
};


// パスの設定
const filePath = {
  entryJs: PATH.join(dir["src"], "agfJs"),
  outputJs: PATH.join(dir["public"], "js"),
};


// ファイル名の設定
const fileName = {
  entryJs: "index.js",
  outputJs: "agf.js",
};


// jsコンパイル設定
const jsSetting = {
  test: /\.js$/,
  exclude: /node_modules/,
  use: [
    {
      loader: "babel-loader",
      options: {
        presets: [["@babel/env"]],
      },
    },
  ],
};


// cssコンパイル設定
const cssSetting = {
  //test → ファイル名を検知する（CSSファイルを検知する）
  test: /\.css/,
  //CSSファイルを検知した場合、以下ルールを適用する（css-loaderを実行せよ）
  use: ["style-loader", "css-loader"]
}


/**
 * 設定
 */
const config = {
  mode: mode,
  devtool: devtool,
  entry: [
    "@babel/polyfill",
    PATH.join(filePath["entryJs"], fileName["entryJs"]),
  ],
  output: {
    path: filePath["outputJs"],
    filename: fileName["outputJs"],
    publicPath: filePath["outputJs"],
  },
  devServer: {
    contentBase: PATH.join(__dirname, "public"),
    open: false, // 起動時にブラウザを開くか開かないか
    watchContentBase: true,
    port: 8000,
    devMiddleware: {
      writeToDisk: true,
    }
  },
  optimization: {
    minimizer: [new terserPlugin({
      extractComments: false, // @see https://qiita.com/TAO_TAO_/items/72a35bd0fedabf4292b2
      terserOptions: {
        compress: {
          drop_console: mode == "production",
        },
      },
    })],
  },
  plugins: [
  ],
  module: {
    rules: [
      jsSetting,
      cssSetting
    ],
  },
  stats: {
    children: true, // エラーの詳細を見るための設定
  },
  target: 'web', // ホットリロードを有効にするための設定
  watch: watch
};


module.exports = [config];
