/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */

import {
	PanelBody,
	TextControl,
	Button,
	__experimentalText as Text
} from '@wordpress/components';
import {
	InnerBlocks,
	useBlockProps,
	InspectorControls
} from '@wordpress/block-editor';
import { useDispatch, useSelect } from '@wordpress/data';
import { store as noticesStore } from '@wordpress/notices';
import apiFetch from '@wordpress/api-fetch';


/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes, clientId}) {
	const parentBlockId = useSelect((select) => {
        // getBlockParents関数を使用して親ブロックのIDの配列を取得
        const { getBlockParents } = select('core/block-editor');
        const parents = getBlockParents(clientId);
        // 最も近い親ブロックのIDを返す（配列の最後の要素）
        return parents[parents.length - 1] || null;
    }, [clientId]);

    const parentBlock = useSelect((select) => {
        // 親ブロックのIDがnullでない場合に限り、親ブロックの情報を取得
        if (!parentBlockId) return null;
        return select('core/block-editor').getBlock(parentBlockId);
    }, [parentBlockId]);


	// 親ブロックが自分自身の場合、エラーメッセージを表示
	if (parentBlock && parentBlock.name === 'agfs/form-wrap') {
		return <p>このブロックの中に同じブロックを入れることはできません。</p>;
	};


	const { actionUrl, prefix, properties, thanksPageUrl } = attributes;
	const { createNotice } = useDispatch( noticesStore );

	/**
	 * 値を保存
	 * @param {string} att - 保存対象のkey
	 * @param {any} val - 保存対象のvalue
	 * @returns void
	 */
	const onChangeEvent = ( att, val ) => {
		setAttributes( { [att]: val } )
	};

	/**
	 * googleフォームの接続情報を取得
	 * @returns void
	 */
	const getFormInfo = async () => {
		if(!actionUrl) {
			createNotice('error' , `googleフォームのURLを入力してください`);
			return;
		}
		try {
			const res = await apiFetch({
				path: '/agfs/v1/googleFormSetting',
				method: 'POST',
				data: {
					url: actionUrl
				}
			});
			setProperties(res);
			prefixControl();
		} catch (error) {
			// エラー処理が必要な場合はここに記述
			console.error("getFormInfo error");
			console.log(error);
			createNotice('error' , `データの取得に失敗しました。`);
		}
	}

	/**
	 * googleフォームの接続情報を保存
	 * @param {properties} data 保存するデータ
	 */
	const setProperties = (data) => {
		const dataset = {};
		dataset["formActionUel"] = data.form_action_uel;
		dataset["labels"] = data.labels;
		setAttributes( { properties: dataset } )
	}

	/**
	 * プレフックスを自動で制作する
	 * @returns void
	 */
	const prefixControl = () => {
		// if(prefix) return;
		console.log("prefixControl");
		setAttributes( {prefix: 'gf' + Date.now()} );
	}

	return (
		<>
			<InspectorControls>
				<PanelBody title={__('google forms URL', 'advancedgforms')}>
					<TextControl
						label="フォームURLを入力"
						value={actionUrl}
						onChange={ ((value) => onChangeEvent('actionUrl', value)) }
					/>
					<Button
						variant="secondary"
						onClick={getFormInfo}
					>フォームを連携する</Button>
				</PanelBody>
				<PanelBody title={'サンクスページURL'}>
					<TextControl
						label="サンクスページのURL入力"
						value={thanksPageUrl}
						onChange={ ((value) => onChangeEvent('thanksPageUrl', value)) }
					/>
				</PanelBody>
				<PanelBody title={'googleフォームフィールドデータ'}>
					{
						(() => {
							if(!properties?.labels) {
								console.log("labelsが見つかりません");
								return <Text>labelsが見つかりません</Text>;
							};

							return (
								<table className='agf-inspector-table'>
									 <thead className='agf-inspector-table__head'>
										<tr>
											<th scope="col">Label</th>
											<th scope="col">ID</th>
										</tr>
									</thead>
									<tbody className='agf-inspector-table__body'>
										{
											properties.labels.map(item => (
												<tr className='table__row'>
													<td>{item.label.replace(/"|'/g, '')}</td>
													<td>{item.id}</td>
												</tr>
											))
										}
									</tbody>
								</table>
							)
						})()
					}
				</PanelBody>
				<PanelBody title={'フォームプレフィックス'} initialOpen={ false }>
					<div class="agf-input -size-s">
						{(prefix) ? prefix : prefixControl()}
					</div>
				</PanelBody>
			</InspectorControls>
			<div { ...useBlockProps() }>
				<InnerBlocks
					templateLock={false}
				/>
			</div>
		</>

	);
}
