/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';

import SaveFieldTemp from '../../components/SaveFieldTemp';

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @return {Element} Element to render.
 */
export default function save({attributes}) {
	const {
		prefix,
		placeholder,
		maxLength,
		rowNum,
		property,
		isRequired
	} = attributes;

	const attr = {};
	const labelFor = `${property}-input`;
	const describedby = `${property}-describedby`;

	if(maxLength) attr["maxLength"] = maxLength;

	const inputAttr = {
		[`${prefix}-item`]: "true",
		id: labelFor,
		'aria-describedby': describedby
	};

	if(isRequired) {
		inputAttr["required"] = true;
		inputAttr["aria-required"] = "true";
	};

	return (
		<SaveFieldTemp
			attributes={attributes}
			htmlDataAttr={attr}
			type="textarea"
			labelFor ={labelFor}
		>
			<>
				<textarea
					className="text-input field-layout__textarea"
					{...inputAttr}
					placeholder={(placeholder == "placeholder") ? false : placeholder}
					rows={Number(rowNum)}

				></textarea>
			</>
		</SaveFieldTemp>
	);
}
