/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import {
	useBlockProps,
	InspectorControls,
	RichText,
} from '@wordpress/block-editor';
import {
	PanelBody,
	CheckboxControl,
	RadioControl,
	TextControl,
	TextareaControl
} from '@wordpress/components';



import FieldTemp from '../../components/FieldTemp';
import LayoutPattern from '../../components/LayoutPattern';
import FiledConnecter from '../../components/FiledConnecter';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes, context }) {
	const contextProperties = context["agfs/properties"];
	const prefix = context["agfs/prefix"];
	const {
		property,
		placeholder,
		layout,
		isRequired,
		maxLength,
		rowNum
	} = attributes;

	setAttributes({ prefix: prefix});

	const requiredCheckboxData = {
		label: "必項",
		slug: "required",
		isChecked: isRequired
	};


	/**
	 * 値を保存
	 * @param {string} att - 保存対象のkey
	 * @param {any} val - 保存対象のvalue
	 * @returns void
	 */
		const onChangeEvent = ( att, val ) => {
			setAttributes( { [att]: val } )
		};

	return (
		<>
			<InspectorControls>
				<FiledConnecter
					contextProperties={contextProperties}
					property={property}
					onChangeEvent={onChangeEvent}
				/>
				<PanelBody title={'オプションの設定'}>
					<CheckboxControl
						label={requiredCheckboxData.label}
						checked={ requiredCheckboxData.isChecked }
						onChange={ ((value) => onChangeEvent('isRequired', value)) }
					/>
					<TextControl
						label="行数"
						help="一般的には7を設定します"
						value={rowNum}
						onChange={ ((value) => onChangeEvent('rowNum', value)) }
					/>
					<TextControl
						label="max文字数"
						help="バイト数でのカウントをしますので半角を1文字、全角を2文字とカウントします"
						value={maxLength}
						onChange={ ((value) => onChangeEvent('maxLength', value)) }
					/>
				</PanelBody>
				<LayoutPattern
					layout={layout}
					onChangeEvent={onChangeEvent}
				/>
			</InspectorControls>
			<FieldTemp
				useBlockProps={useBlockProps}
				attributes={attributes}
				onChangeEvent={onChangeEvent}
			>
				<TextareaControl
					value={placeholder}
					rows={Number(rowNum)}
					onChange={ ((value) => onChangeEvent('placeholder', value)) }
				/>
			</FieldTemp>
		</>
	);
}
