/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';

import SaveFieldTemp from '../../components/SaveFieldTemp';

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @return {Element} Element to render.
 */
export default function save({attributes}) {
	const {
		prefix,
		property,
		msg,
		label,
		layout,
        isRequired,
		option,
		placeholder,
		Z2HandH2Z,
		addressLabel
	} = attributes;


	const labelFor = `${property}-input`;
	const describedby = `${property}-describedby`;


	const setting = {
		className: `from__field field-layout -${layout}`,
		[`${prefix}-type`]: "text",
	};

	const inputAttr = {
		[`${prefix}-item`]: "true",
		id: labelFor,
		'aria-describedby': describedby,
		placeholder: (placeholder == "placeholder") ? false : placeholder
	};

	if(isRequired) {
		inputAttr["required"] = true;
		inputAttr["aria-required"] = "true";
	};

	if(isRequired) setting["required"] = true;

	if(Z2HandH2Z) setting[Z2HandH2Z] = "true";

	if(option) setting[option] = "true";

	if(option == "zip") setting["zip"] = addressLabel;

	if(option == "address") setting["address"] = addressLabel;

	if(option == "time") {
		inputAttr["format"] = "H:i";
		setting[`${prefix}-type`] = "time";
	}

	if(option == "calendar") {
		inputAttr["format"] = "Y-m-d";
		setting[`${prefix}-type`] = "calendar";
	}


	/**
	 * inputのtypeの値を取得する
	 * @returns {string}
	 */
	const getType = () => {
		if(option == "number") return "number";
		if(
			option == "price" ||
			option == "phone" ||
			option == "zip" ||
			option == "calendar"
		) return "tel";
		if(option == "time") return "time";
		if(option == "email") return "email";
		return "text";
	}


	/**
	 * autocomplete属性の値を取得する
	 * @returns {string}
	 * @see https://developer.mozilla.org/ja/docs/Web/HTML/Attributes/autocomplete
	 */
	const getAutocomplete = () => {
		if(option == "email") return "email";
		if(option == "zip") return "postal-code";
		if(option == "phone") return "tel";
		if(option == "state") return "address-level1"; // 都道府県レベルの住所
		if(option == "address") return "address-level2"; // 市区町村レベルの住所
	}


	return (
		<SaveFieldTemp
			attributes={attributes}
			htmlDataAttr={setting}
			type="textarea"
			labelFor ={labelFor}
		>
			<input
				className="text-input field-layout__input-item"
				{...inputAttr}
				type={getType()}
				autocomplete={getAutocomplete()}
			/>
		</SaveFieldTemp>
	)
}


// return (
// 	<div
// 		{...useBlockProps.save(setting)}
// 	>
// 		<input
// 			type="hidden"
// 			name={property}
// 			value=""
// 			{...{[`${prefix}-input`]: "true"}}
// 		/>
// 		{
// 			layout == "pattern5"
// 			?<div className="field-layout__label-area">
// 				<div>
// 					<div className="field-layout__label agf-flex -gap12 -y-start">
// 						<label htmlFor={labelFor}>{label}</label>
// 						{
// 							isRequired && <div className='field-layout__label-required'>必項</div>
// 						}
// 					</div>
// 					<div className="field-layout__msg" id={describedby}>{msg}</div>
// 				</div>
// 			</div>
// 			:<>
// 				<div className={`field-layout__label-area`}>
// 					<label className="field-layout__label" htmlFor={labelFor}>{label}</label>
// 					{
// 						isRequired && <div className='field-layout__label-required'>必項</div>
// 					}
// 				</div>
// 				<div className={`field-layout__msg ${(layout == "pattern4") ? 'agf-!margin0' : ''}`} id={describedby}>{msg}</div>
// 			</>
// 		}
// 		<div className={`field-layout__input-wrap ${(layout == "pattern3" || layout == "pattern5") ? 'agf-!margin0' : ''}`}>
// 			<input
// 				className="text-input field-layout__input-item"
// 				{...inputAttr}
// 				type={getType()}
// 				autocomplete={getAutocomplete()}
// 			/>
// 		</div>
// 		<div {...{[`${prefix}-error`]: "true"}} className="field-layout__error" role="alert"></div>
// 	</div>
// )