/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import {
	PanelBody,
	CheckboxControl,
	RadioControl,
	TextControl
} from '@wordpress/components';
import {
	useBlockProps,
	InspectorControls,
	RichText,
} from '@wordpress/block-editor';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

import FieldTemp from '../../components/FieldTemp';
import LayoutPattern from '../../components/LayoutPattern';
import FiledConnecter from '../../components/FiledConnecter';


/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes, context }) {
	const contextProperties = context["agfs/properties"];
	const prefix = context["agfs/prefix"];
	const {
		property,
		placeholder,
		layout,
		isRequired,
		option,
		Z2HandH2Z,
		addressLabel
	} = attributes;

	setAttributes({ prefix: prefix});

	/**
	 * オプション
	 * TODO: 後に共通化
	 */
	const optionList = [
		{
			label: "オプションなし",
			value: "",
		},
		{
			label: "数字のみ",
			value: "number",
		},
		{
			label: "金額",
			value: "price",
		},
		{
			label: "電話",
			value: "phone",
		},
		{
			label: "メール",
			value: "email",
		},
		{
			label: "郵便番号",
			value: "zip",
		},
		{
			label: "住所入力",
			value: "address",
		},
		{
			label: "時間(H:i)",
			value: "time",
		},
		{
			label: "カレンダー(JIS規格: YYYY-MM-DD)",
			value: "calendar",
		},
	]

	const requiredCheckboxData = {
		label: "必項",
		slug: "required",
		isChecked: isRequired
	};

	const Z2HandH2ZData = [
		{
			label: "変換なし",
			value: "",
		},
		{
			label: "半角を全角へ変換します",
			value: "h2z",
		},
		{
			label: "全角を半角へ変換します",
			value: "z2h",
		}
	];

	/**
	 * 値を保存
	 * @param {string} att - 保存対象のkey
	 * @param {any} val - 保存対象のvalue
	 * @returns void
	 */
	const onChangeEvent = ( att, val ) => {
		setAttributes( { [att]: val } )
	};

	return (
		<>
			<InspectorControls>
				<FiledConnecter
					contextProperties={contextProperties}
					property={property}
					onChangeEvent={onChangeEvent}
				/>
				<PanelBody title={'オプションの設定'}>
					<CheckboxControl
						label={requiredCheckboxData.label}
						checked={ requiredCheckboxData.isChecked }
						onChange={ ((value) => onChangeEvent('isRequired', value)) }
					/>
					<RadioControl
						label="半角/全角変換"
						selected={Z2HandH2Z}
						onChange={ ((value) => onChangeEvent('Z2HandH2Z', value)) }
						options={Z2HandH2ZData}
					/>
					<RadioControl
						label="オプションを選択"
						selected={option}
						onChange={ ((value) => onChangeEvent('option', value)) }
						options={optionList}
					/>
					{
						(option == "zip" || option == "address") ? <TextControl
							label="連携用idを入力"
							value={addressLabel}
							onChange={ ((value) => onChangeEvent('addressLabel', value)) }
						/> : ""
					}
				</PanelBody>
				<LayoutPattern
					layout={layout}
					onChangeEvent={onChangeEvent}
				/>
			</InspectorControls>
			<FieldTemp
				useBlockProps={useBlockProps}
				attributes={attributes}
				onChangeEvent={onChangeEvent}
			>
				<RichText
					tagName="div"
					value={placeholder}
					onChange={ ((value) => onChangeEvent('placeholder', value)) }
					placeholder={
						'プレースホルダーを入力してください'
					}
					className="field-layout__input-item"
				/>
			</FieldTemp>
		</>
	);
}
