/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import {
	PanelBody,
	CheckboxControl,
	RadioControl,
} from '@wordpress/components';
import {
	useBlockProps,
	InspectorControls,
	RichText,
} from '@wordpress/block-editor';


/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes, context }) {
	const prefix = context["agfs/prefix"];
	const {
		label,
		isDummy,
		honeySpotMsg
	} = attributes;

	setAttributes({ prefix: prefix});

	const isDummyCheckboxData = {
		label: "ハニースポットを有効にする",
		slug: "isDummy",
		isChecked: isDummy
	};

	/**
	 * 値を保存
	 * @param {string} att - 保存対象のkey
	 * @param {any} val - 保存対象のvalue
	 * @returns void
	 */
		const onChangeEvent = ( att, val ) => {
			setAttributes( { [att]: val } )
		};

	return (
		<>
			<InspectorControls>
				<PanelBody title={'オプションの設定'}>
					<CheckboxControl
						label={isDummyCheckboxData.label}
						checked={ isDummyCheckboxData.isChecked }
						onChange={ ((value) => onChangeEvent('isDummy', value)) }
					/>
				</PanelBody>
			</InspectorControls>
			<div {...useBlockProps()}>
				{
					isDummy && <div className='agf-flex -x-center'>
									<div>
										<input type="checkbox" value="" name={`${prefix}-required`} id={`${prefix}-required`}/>
										<RichText
											tagName="label"
											value={honeySpotMsg}
											onChange={ ((value) => onChangeEvent('honeySpotMsg', value)) }
											placeholder={'あなたが人間の場合、こちらにチェックを入れないでください。'}
										/>
										{/* <label htmlFor={`${prefix}-required`}></label> */}
									</div>
								</div>
				}
				<div className='agf-flex -x-center u-mt-16'>
					<div className='agf-submit-btn'>
						<RichText
							tagName="span"
							value={label}
							onChange={ ((value) => onChangeEvent('label', value)) }
							placeholder={'ボタンのラベル(例:送信'}
							className="agf-submit-btn__label"
						/>
					</div>
				</div>
			</div>
		</>

	);
}
