/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import {
	SelectControl,
	PanelBody,
	CheckboxControl,
	TextControl
} from '@wordpress/components';
import {
	useBlockProps,
	InspectorControls
} from '@wordpress/block-editor';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';


import FieldTemp from '../../components/FieldTemp';
import LayoutPattern from '../../components/LayoutPattern';
import FiledConnecter from '../../components/FiledConnecter';
import OptionsController from '../../components/OptionsController';


/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes, context }) {
	const contextProperties = context["agfs/properties"];
	const prefix = context["agfs/prefix"];

	const {
		property,
		layout,
		selectBoxOptions,
		selected,
		isRequired,
		state,
		stateId
	} = attributes;

	setAttributes({ prefix: prefix});

	const requiredCheckboxData = {
		label: "必項",
		slug: "required",
		isChecked: isRequired
	};

	/**
	 * 値を保存
	 * @param {string} att - 保存対象のkey
	 * @param {any} val - 保存対象のvalue
	 * @returns void
	 */
	const onChangeEvent = ( att, val ) => {
		setAttributes( { [att]: val } )
	};

	return (
		<>
			<InspectorControls>
				<FiledConnecter
					contextProperties={contextProperties}
					property={property}
					onChangeEvent={onChangeEvent}
				/>
				<PanelBody title={'オプションの設定'}>
					<CheckboxControl
						label={requiredCheckboxData.label}
						checked={ requiredCheckboxData.isChecked }
						onChange={ ((value) => onChangeEvent('isRequired', value)) }
					/>
					<CheckboxControl
						label="都道府県"
						checked={ state }
						onChange={ ((value) => onChangeEvent('state', value)) }
					/>
					{
						state && <TextControl
							label="連携用idを入力"
							value={stateId}
							onChange={ ((value) => onChangeEvent('stateId', value)) }
						/>
					}
					{
						!state && <OptionsController
							title={"セレクトボックス選択の追加"}
							options={selectBoxOptions}
							setAttributesHasKey={((value) => {
								setAttributes({selectBoxOptions: value})
							})}
						/>
					}
				</PanelBody>
				<LayoutPattern
					layout={layout}
					onChangeEvent={onChangeEvent}
				/>
			</InspectorControls>
			<FieldTemp
				useBlockProps={useBlockProps}
				attributes={attributes}
				onChangeEvent={onChangeEvent}
			>
				{
					state &&
					<select className="field-layout__input-select">
						<option>都道府県一覧が表示されます</option>
					</select>
				}
				{
					!state && (() => {
						return(
							<select
								className="field-layout__input-select"
								onChange={ ((value) => onChangeEvent('selected', value)) }
								value={selected}
							>
								{
									selectBoxOptions.map(item => (
										<option value={item.value}>{item.label}</option>
									))
								}
							</select>
						)
					})()
				}
			</FieldTemp>
		</>

	);
}
