/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';

import SaveFieldTemp from '../../components/SaveFieldTemp';

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @return {Element} Element to render.
 */
export default function save({attributes}) {
	const {
		prefix,
		radioOptions,
		property,
		isRequired
	} = attributes;

	const describedby = `${property}-describedby`;

	const labelFor = `${property}-group-label`;

	const inputAttr = {
		[`${prefix}-item`]: "true",
	};

	if(isRequired) {
		inputAttr["required"] = true;
		inputAttr["aria-required"] = "true";
	};

	return (
		<SaveFieldTemp
			attributes={attributes}
			type="radio"
			labelFor={labelFor}
		>
			<div
				role="group"
				className="field-layout__input-radio"
				aria-describedby={describedby}
				aria-labelledby={labelFor}
			>
				{
					radioOptions.map((item) => (
						<div className='agf-radio'>
							<input
								className='agf-radio__input'
								type="radio"
								value={item.value}
								{...inputAttr}
								name={`${property}-radio`}
								id={`${property}-${item.value}`}
							/>
							<label
								className='agf-radio__label'
								htmlFor={`${property}-${item.value}`}
							>{item.label}</label>
						</div>
					))
				}
			</div>
		</SaveFieldTemp>
	);
}
