/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import {
	PanelBody,
	Button,
	CheckboxControl
} from '@wordpress/components';
import {
	useBlockProps,
	InspectorControls,
	RichText
} from '@wordpress/block-editor';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';


import FieldTemp from '../../components/FieldTemp';
import LayoutPattern from '../../components/LayoutPattern';
import FiledConnecter from '../../components/FiledConnecter';
import OptionsController from '../../components/OptionsController';


/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes, context }) {
	const contextProperties = context["agfs/properties"];
	const prefix = context["agfs/prefix"];

	const {
		property,
		layout,
		radioOptions,
		selected,
		isRequired
	} = attributes;

	setAttributes({ prefix: prefix});

	const requiredCheckboxData = {
		label: "必項",
		slug: "required",
		isChecked: isRequired
	};

	/**
	 * 値を保存
	 * @param {string} att - 保存対象のkey
	 * @param {any} val - 保存対象のvalue
	 * @returns void
	 */
	const onChangeEvent = ( att, val ) => {
		setAttributes( { [att]: val } )
	};

	const radioOptionsOnChangeEvent = ( att, val, index ) => {
		const newData = JSON.parse(JSON.stringify(radioOptions))
		newData[index][att] = val
		setAttributes( { radioOptions: newData } )
	}

	const addRadioOptions = () => {
		const newData = JSON.parse(JSON.stringify(radioOptions))
		newData.push({
			label: "",
			value: ""
		})
		setAttributes( { radioOptions: newData } )
	}

	if(radioOptions.length <= 0) {
		addRadioOptions()
	}

	const selectedChangeEvent = (value) => {
		setAttributes( { selected: value.target.value } )
	}

	return (
		<>
			<InspectorControls>
				<FiledConnecter
					contextProperties={contextProperties}
					property={property}
					onChangeEvent={onChangeEvent}
				/>
				<PanelBody title={'オプションの設定'}>
					<CheckboxControl
						label={requiredCheckboxData.label}
						checked={ requiredCheckboxData.isChecked }
						onChange={ ((value) => onChangeEvent('isRequired', value)) }
					/>
					<OptionsController
						title={"ラジオボタンの選択肢"}
						options={radioOptions}
						setAttributesHasKey={((value) => {
							setAttributes({radioOptions: value})
						})}
					/>
				</PanelBody>
				<LayoutPattern
					layout={layout}
					onChangeEvent={onChangeEvent}
				/>
			</InspectorControls>
			<FieldTemp
				useBlockProps={useBlockProps}
				attributes={attributes}
				onChangeEvent={onChangeEvent}
			>
				<div className="field-layout__input-radio">
					{
						radioOptions.map((item) => (
							<div className='agf-radio'>
								<input
									className='agf-radio__input'
									type="radio"
									name={`${property}-radio`}
									id={`${property}-${item.value}`}
									value={item.value}
									checked={item.value === selected}
									onChange={selectedChangeEvent}
								/>
								<label
									className='agf-radio__label'
									htmlFor={`${property}-${item.value}`}
								>{item.label}</label>
							</div>
						))
					}
				</div>
			</FieldTemp>
		</>

	);
}
