/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import {
	PanelBody,
	Button,
	CheckboxControl,
	Modal
} from '@wordpress/components';
import {
	useBlockProps,
	InspectorControls,
	RichText
} from '@wordpress/block-editor';

import { useState } from '@wordpress/element';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';


import FieldTemp from '../../components/FieldTemp';
import LayoutPattern from '../../components/LayoutPattern';
import FiledConnecter from '../../components/FiledConnecter';
import OptionsController from '../../components/OptionsController';


/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes, context }) {
	const contextProperties = context["agfs/properties"];
	const prefix = context["agfs/prefix"];

	const {
		property,
		layout,
		checkboxOptions,
		selected,
		isRequired
	} = attributes;

	const [isOpen, setOpen] = useState(false);

	setAttributes({ prefix: prefix});

	const requiredCheckboxData = {
		label: "必項",
		slug: "required",
		isChecked: isRequired
	};

	/**
	 * 値を保存
	 * @param {string} att - 保存対象のkey
	 * @param {any} val - 保存対象のvalue
	 * @returns void
	 */
	const onChangeEvent = ( att, val ) => {
		setAttributes( { [att]: val } )
	};

	const checkboxOptionsOnChangeEvent = ( att, val, index ) => {
		const newData = JSON.parse(JSON.stringify(checkboxOptions))
		newData[index][att] = val
		setAttributes( { checkboxOptions: newData } )
	}

	const addCheckboxOptions = () => {
		const newData = JSON.parse(JSON.stringify(checkboxOptions))
		newData.push({
			label: "",
			value: ""
		})
		setAttributes( { checkboxOptions: newData } )
	}

	if(checkboxOptions.length <= 0) {
		addCheckboxOptions()
	}

	const selectedChangeEvent = (value) => {
		if(selected.includes(value.target.value)) {
			setAttributes( { selected:  selected.filter((item) => item != value.target.value)} )
		} else {
			const newSelected = [...selected]
			newSelected.push(value.target.value);
			setAttributes( { selected:  newSelected} )
		}
	}

	return (
		<>
			<InspectorControls>
				<FiledConnecter
					contextProperties={contextProperties}
					property={property}
					onChangeEvent={onChangeEvent}
				/>
				<PanelBody title={'オプションの設定'}>
					<CheckboxControl
						label={requiredCheckboxData.label}
						checked={ requiredCheckboxData.isChecked }
						onChange={ ((value) => onChangeEvent('isRequired', value)) }
					/>
					<OptionsController
						options={checkboxOptions}
						setAttributesHasKey={((value) => {
							setAttributes({checkboxOptions: value})
						})}
					/>
				</PanelBody>
				<LayoutPattern
					layout={layout}
					onChangeEvent={onChangeEvent}
				/>
			</InspectorControls>
			<FieldTemp
				useBlockProps={useBlockProps}
				attributes={attributes}
				onChangeEvent={onChangeEvent}
			>
				<div className="field-layout__input-checkbox">
					{
						checkboxOptions.map((item) => (
							<div className='agf-radio'>
								<input
									className='agf-radio__input'
									type="checkbox"
									name={`${property}-checkbox`}
									id={`${property}-${item.value}`}
									value={item.value}
									checked={selected.includes(item.value)}
									onChange={selectedChangeEvent}
								/>
								<label
									className='agf-radio__label'
									htmlFor={`${property}-${item.value}`}
								>{item.label}</label>
							</div>
						))
					}
				</div>
			</FieldTemp>
		</>
	);
}
