// @ts-check
const fs = require("fs");

/**
 * @see sass : https://sass-lang.com/documentation/js-api
 * @see sass-options : https://sass-lang.com/documentation/cli/dart-sass
 */
// import sass from 'sass';
// import autoprefixer from 'autoprefixer';
// import postcss from 'postcss';
// import csso from 'csso';
// import mqpacker from 'css-mqpacker';

const sass = require("sass");
const autoprefixer = require("autoprefixer");
const postcss = require("postcss");
const csso = require("csso");
const mqpacker = require("css-mqpacker");

const MODE = 'development'; // development or production

/**
 * sassファイルのパスを受け取ってcssの文字列を返す
 * @param {string} filePath
 * @returns {string} CSS
 */
const processSass = (filePath) => {
  try {
    let sassResult = sass.compile(filePath);
    return sassResult.css.toString();
  } catch (e) {
    console.error(e);
  }

  return '';
};

/**
 * 文字列で受け取ったCSSにPostCSSを適用する
 * @param {string} css
 * @returns {string}
 */
const processPostCss = (css) => {
  let postCssResult = null;
  let plugins = [mqpacker(), autoprefixer()];

  try {
    postCssResult = postcss(plugins).process(css);
  } catch (e) {
    console.error(e);
  }

  return postCssResult.css;
};

/**
 * 文字列で受け取ったCSSにcssoを適用する
 * @param {string} css
 */
const processCSSO = (css, minify) => {
  if(minify === false) return css;
  return csso.minify(css).css;
};

/**
 * すべてを実行する
 * @param {string} filePath
 * @returns {string}
 */
function all(filePath, minify) {
  console.log(`[INFO] start ${filePath} transpiling...`);
  const compiledCSS = processSass(filePath);
  const afterPostCSS = processPostCss(compiledCSS);
  const afterCSSO = processCSSO(afterPostCSS, minify);

  return afterCSSO;
};

/**
 * メインのCSS（テーマ直下のstyle.css）を作る
 */
function style(config = [], minify = false) {
  if(config.length == 0) return;

  for(let i = 0; config.length > i; i++) {
    if(
      config[i].hasOwnProperty('entryPath') &&
      config[i].entryPath !== '' &&
      config[i].hasOwnProperty('outputPath') &&
      config[i].outputPath !== ''
    ) {
      const input = config[i].entryPath;
      const result = all(input, minify);
      let outputPath = config[i].outputPath.split('/');
      const outputFileName = outputPath.pop();
      const outputDir = outputPath.join('/');

      fs.mkdir(outputDir, { recursive: true },  (err)=> {
        if (err) { throw err; }
        fs.writeFileSync(outputDir + '/' + outputFileName, result);
      });
    };
  }
}

module.exports = style;


