import flatpickr from 'flatpickr/dist/flatpickr.min.js';
import { Japanese } from "flatpickr/dist/l10n/ja.js";
import 'flatpickr/dist/flatpickr.min.css';

import SetState from "./SetState.js";
import Debugger from "./Debugger.js";

const zpicloudEndPoint = "https://zipcloud.ibsnet.co.jp/api/search";


function escapeHtml (string) {
    if(typeof string !== 'string') {
      return string;
    }
    return string.replace(/[&'`"<>]/g, function(match) {
      return {
        '&': '&amp;',
        "'": '&#x27;',
        '`': '&#x60;',
        '"': '&quot;',
        '<': '&lt;',
        '>': '&gt;',
      }[match]
    });
}

export default class InputEl {

    constructor(data) {
        this.data = data;
    }


    start() {
        if(this.data.options.state) new SetState(this.data.el);
        if(this.data.type == "textarea") this.textCounter();
        this.setEvent();
    }

    /**
     * 数字を金額へ変換する処理
     */
    num2price(str) {
        Debugger.log("num2price event");
        // ０〜９（全角数字）を見つけて半角数字に変換
        str = str.replace(/[０-９]/g,s => String.fromCharCode(s.charCodeAt(0) - 65248));
        // 数字以外を削除
        str = str.replace(/\D/g,'');
        // 3桁カンマ区切りへ変換
        let price = Number(str).toLocaleString();
        return (price == 0) ? "" : price;
    }

    /**
     * 3桁カンマを数字に変換
     */
    price2num(str) {
        Debugger.log("price2num event");
        return str.replace(/,/g, "");
    }

    /**
     * 電話番号から全角を取り除き、数字の全角は半角へ変換する処理
     */
    z2phone(str) {
        return str.replace(/[０-９]/g, (s) => String.fromCharCode(s.charCodeAt(0) - 65248)).replace(/[ー−]/g,'-').replace(/[^\-\d]/g,'');
    }

    /**
     * 全角から半角へ変換する
     */
    z2h(str) {
        return str.replace(/[Ａ-Ｚａ-ｚ０-９]/g, function(s) {
            return String.fromCharCode(s.charCodeAt(0) - 0xFEE0);
        });
    }

    /**
     * 半角から全角へ変換する
     */
    h2z(str) {
        return str.replace(/[A-Za-z0-9]/g, function(s) {
            return String.fromCharCode(s.charCodeAt(0) + 0xFEE0);
        });
    }

    /**
     * カタカナ => ひらがな 変換
     */
    kana2Hira(str) {
        return str.replace(/[\u30a1-\u30f6]/g, function(s){
            return String.fromCharCode(s.charCodeAt(0) - 0x60);
        });
    }

    /**
     * ひらがな -> カタカナ 変換
     */
    hira2Kana(str) {
        return str.replace(/[\u3041-\u3096]/g, function(s){
            return String.fromCharCode(s.charCodeAt(0) + 0x60);
        });
    }

    /**
     * バリデーションチェックを行い、ステータスを変更します。
     */
    check() {
        Debugger.log("!!check event!!");

        if(this.data.type == "required") {
            if(this.data.el.checked) {
                Debugger.log("ボット error");
                this.data.check.status = false;
                this.data.check.msg = "※チェックを外してください";
                return false;
            } else {
                this.data.check.msg = "";
            }
        }

        if(this.data.options.required && (this.data.type == "checkbox" || this.data.type == "radio")) {
            const values = [];
            this.data.el.forEach((item) => {
                if(item.checked) values.push(item.value);
            })

            if(values.length <= 0) {
                this.data.check.msg = "入力してください";
                this.data.check.status = false;
                return false;
            }
        }

        if(this.data.options.required && this.data.el.value == "") {
            this.data.check.msg = "入力してください";
            this.data.check.status = false;
            return false;
        }

        if(this.data.type == "textarea" && this.data.options.maxlength) {
            const num = this.getTextSize(this.data.el.value);
            if(num > this.data.options.maxlength) {
                this.data.check.msg = "制限文字数を超えています";
                this.data.check.status = false;
                return false;
            } else {
                this.data.check.msg = "";
                this.data.check.status = true;
                return true;
            }
        }


        if(this.data.type == "submit") {
            this.data.check.status = false;
            return false;
        }
        if(this.data.options.email) {
            if(!this.data.el.value.match(/.+@.+\..+/)) {
                Debugger.log("メールアドレスの形式ではありません");
                this.data.check.msg = "メールアドレスの形式ではありません";
                this.data.check.status = false;
                return false;
            }
            this.data.check.msg = "";
            this.data.check.status = true;
            return true;
        }
        if(this.data.options.phone) {
            const tel = this.data.el.value.replace(/[━.*‐.*―.*－.*\-.*ー.*\-]/gi,'')
            if(!tel.match(/^(0[5-9]0[0-9]{8}|0[1-9][1-9][0-9]{7})$/)) {
                Debugger.log("電話番号の形式ではありません");
                this.data.check.msg = "電話番号の形式ではありません";
                this.data.check.status = false;
                return false;
            }
            this.data.check.msg = "";
            this.data.check.status = true;
            return true;
        }
        this.data.check.msg = "";
        this.data.check.status = true;
        return true;
    }

    /**
     * バリデーション結果をレンダリングする
     */
    renderValid() {
        this.renderMsg();
        this.renderStatus();
        this.setAriaInvalid();
    }

    /**
     * エラーメッセージやサクセスメッセージをレンダリングします
     */
    renderMsg() {
        if(!this.data.elForMsg) return;
        Debugger.log("renderMsg event");
        Debugger.log(this.data.elForMsg);
        this.data.elForMsg.innerText = this.data.check.msg;
    }

    /**
     * ステータスをレンダリング
     */
    renderStatus() {
        Debugger.log("renderStatus event");

        if(this.data.type == "radio" || this.data.type == "checkbox"){
            this.data.el.forEach((el) => {
                if(this.data.check.status) {
                    el.classList.add("-success");
                    el.classList.remove("-error");
                } else {
                    el.classList.add("-error");
                    el.classList.remove("-success");
                }
            })
            return;
        }

        if(this.data.check.status) {
            this.data.el.classList.add("-success");
            this.data.el.classList.remove("-error");
        } else {
            this.data.el.classList.add("-error");
            this.data.el.classList.remove("-success");
        }
    }


    /**
     * aria-invalidをセットする
     */
    setAriaInvalid() {
        if(this.data.type == "radio" || this.data.type == "checkbox"){
            this.data.el.forEach((el) => {
                el.setAttribute("aria-invalid", !this.data.check.status);
            });

            return;
        }
        this.data.el.setAttribute("aria-invalid", !this.data.check.status);
    }

    /**
     * 送信用inputへ値をセット
     */
    setValueForSendInput() {
        Debugger.log("setValueForSendInput event");

        if(this.data.type == "radio" || this.data.type == "checkbox") {
            const values = [];
            this.data.el.forEach((item) => {
                if(item.checked) values.push(item.value);
            })
            if(this.data.elForSend) this.data.elForSend.value = escapeHtml(values.join(","));

            return;
        } else if(this.data.type == "submit") {
            return;
        };

        if(this.data.elForSend) this.data.elForSend.value = escapeHtml(this.data.el.value);
    }

    /**
     * 入力用inputへ値をセット
     */
    setValue() {
        Debugger.log("setValue event");

        if(this.data.type == "radio" || this.data.type == "checkbox") {
            const values = [];
            this.data.el.forEach((item) => {
                if(item.checked) values.push(item.value);
            })
            this.data.value = escapeHtml(values.join(","));

            return;
        } else if(this.data.type == "submit") {
            return;
        };

        this.data.value = escapeHtml(this.data.el.value);
    }

    /**
     * イベントをセットします
     */
    setEvent() {
        if(this.data.type == "radio") {
            this.data.el.forEach((item) => {
                item.addEventListener("change", () => {
                    Debugger.log("change");
                    this.check();
                    setTimeout(() => {
                        this.renderValid();
                        this.setValue();
                        this.setValueForSendInput();
                    }, 1);
                });
            })
        } else if(this.data.type == "select") {
            this.data.el.addEventListener("change", () => {
                this.optionsEvents();
                this.check();
                setTimeout(() => {
                    this.renderValid();
                    this.setValue();
                    this.setValueForSendInput();
                }, 1);

            });
        } else if(this.data.type == "text" && this.data.options.price) {
            this.data.el.addEventListener("blur", () => {
                this.data.el.value = escapeHtml(this.num2price(this.data.el.value));
                this.optionsEvents();
                this.check();
                setTimeout(() => {
                    this.renderValid();
                    this.setValue();
                    this.setValueForSendInput();
                }, 1);
            });

            this.data.el.addEventListener("focus", () => {
                this.data.el.value = escapeHtml(this.price2num(this.data.el.value));
            });
        }  else if(this.data.type == "text" && this.data.options.zip) {
            // 郵便番号用
            this.data.el.addEventListener("keyup", () => {
                Debugger.log("zip keyup");
                const regex = /^[0-9]{3}-?[0-9]{4}$/;
                const status = regex.test(this.data.el.value);
                if(status) {
                    this.setAutoZip();
                    this.setValue();
                    this.setValueForSendInput();
                }
            });
            this.data.el.addEventListener("blur", () => {
                Debugger.log("zip blur");
                const regex = /^[0-9]{3}-?[0-9]{4}$/;
                const status = regex.test(this.data.el.value);
                this.data.check.msg = (status) ? "" : "郵便番号の形式ではありません";
                setTimeout(() => {
                    this.renderValid();
                    this.setValue();
                    this.setValueForSendInput();
                }, 1);
            });
        } else if(this.data.type == "text") {
            this.data.el.addEventListener("blur", () => {
                this.optionsEvents();
                this.check();
                setTimeout(() => {
                    this.renderValid();
                    this.renderValid();
                    this.setValueForSendInput();
                }, 1);
            });
        } else if(this.data.type == "textarea") {
            this.data.el.addEventListener("blur", () => {
                this.check();
                setTimeout(() => {
                    this.renderValid();
                    this.setValue();
                    this.setValueForSendInput();
                }, 1);
            });
        }  if(this.data.type == "checkbox") {
            this.data.el.forEach((item) => {
                item.addEventListener("change", () => {
                    Debugger.log("change");
                    this.check();
                    setTimeout(() => {
                        this.renderValid();
                        this.setValue();
                        this.setValueForSendInput();
                    }, 1);
                });
            })
        } else if(this.data.type == "time") {
            this.data.el.addEventListener("blur", () => {
                setTimeout(() => {
                    this.check();
                    this.renderValid();
                    this.setValue();
                    this.setValueForSendInput();
                });
            });

        } else if(this.data.type == "calendar") {
            const format = (this.data.el.getAttribute("format")) ? this.data.el.getAttribute("format") : 'Y.m.d（D）';
            new flatpickr(this.data.el, {
                locale : Japanese, // 日本語用モジュールを適用
                dateFormat : format,
                disableMobile: true,
                onClose: (function() {
                    this.check();
                    this.renderValid();
                    this.setValue();
                    this.setValueForSendInput();
                }).bind(this)
            });
        } else if(this.data.type == "required") {
            // ハニースポット用
            this.data.el.addEventListener("change", () => {
                Debugger.log("required change");
                this.check();
                this.renderValid();
            });
        } else if(this.data.type == "submit") {
            // this.data.el.addEventListener("check", (e) => {
            //     Debugger.log("submit");
            // })
        }
    }

    /**
     * デフォルト値をセットする
     * @param {string} value - セットする値
     * 値のパターン
     * text => string
     * radio => string valueの値
     * checkbox => string +区切りのテキスト
     */
    setDefaultValue(value) {
        if(this.data.type == "radio") {
            this.data.el.forEach((item) => {
                if(item.value == value)
                    item.checked = true;
                else
                    item.checked = false;
            })
            this.data.elForSend.value = escapeHtml(value);
            return;
        } else if(this.data.type == "checkbox") {
            const valueArr = value.split(" ");
            this.data.el.forEach((item) => {
                if(valueArr.indexOf(item.value) != -1)
                    item.checked = true;
                else
                    item.checked = false;
            });
            this.data.elForSend.value = escapeHtml(valueArr.join(","));
            return;
        } else if(this.data.type == "submit") {
            return;
        }

        this.data.el.value = escapeHtml(value);
        this.data.elForSend.value = escapeHtml(value);
    }


    getData() {
        return this.data;
    }

    optionsEvents() {
        if(this.data.options.z2h) {
            this.data.el.value = escapeHtml(this.z2h(this.data.el.value));
        }

        if(this.data.options.h2z) {
            this.data.el.value = escapeHtml(this.h2z(this.data.el.value));
        }
    }

    async setAutoZip() {
        Debugger.log("setAutoZip");

        const zipcode = this.data.el.value;
        if(!zipcode) return;

        try {
            const response = await fetch(`${zpicloudEndPoint}?zipcode=${zipcode}`);

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data);
            }

            Debugger.log("search address success");
            Debugger.log(data.results);

            // 住所情報のレンダリング
            if(data.results === null) {
                Debugger.log("address undefined");
                delete this.data.address;
                this.data.check.msg = "正しい郵便番号ではないか、住所が見つかりません。";
                this.data.check.status = false;
                return;
            }

            const address = data.results[0];

            this.data.address = address;
            this.data.check.msg = "";
            this.data.check.status = true;
        } catch (error) {
            Debugger.error("search address catch error");
            Debugger.error(error);
        }
    }

    textCounter() {
        const countEl = document.createElement("div");
        const maxLength = this.data.options.maxlength;

        this.data.el.after(countEl);

        if(Number(maxLength) <= 0) return;

        countEl.innerText = `0/${maxLength}`;

        countEl.classList.add("field-layout__textarea-counter");

        this.data.el.addEventListener("keyup", (event) => {
            const len = this.getTextSize(event.target.value);

            if(len > maxLength) {
                this.data.el.classList.add("-over");
            }

            countEl.innerText = `${len}/${maxLength}`;
        });
    }

    /**
     * テキストサイズをカウントします
     * 半角1文字、全角2文字です
     * @param {string} str - カウントしたい文字
     * @returns {number}
     */
    getTextSize(str) {
        let len = 0;

        for (let i = 0; i < str.length; i++) {
            (str[i].match(/[ -~]/)) ? len += 1 : len += 2;
        };

        return len;
    }
}