<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use DiDom\Document;
use Tholu\Packer\Packer;

class AGFS_App {
    public $option_setting = array();
    public $plugin_slug = "";
    public $plugin_base_name = "";
    public $submitted_script = "";
    public $encrypted_script = "";

	public function __construct() {
		$this->activate_actions();
		$this->activate_filters();

        $this->plugin_slug = 'advancedgforms';
        $this->plugin_base_name = $this->plugin_slug . "/advancedgforms.php";

        $this->option_setting = array(
            'decrypt_key',
            'css_off',
            'primary_color',
            'btn_text_color',
            'font_color'
        );
	}


    /**
	 * アクションを実行
	 *
	 * @return void
	 */
	private function activate_actions() {
        add_action('wp_enqueue_scripts', array( $this, 'add_scripts' ));

        add_action('admin_menu', array( $this, 'add_setting_menu_page' ) );
        add_action('admin_enqueue_scripts', array( $this, 'add_admin_scripts' ));
    }

    /**
	 * フィルターを実行
	 *
	 * @return void
	 */
	private function activate_filters() {
        add_filter( 'the_content', array( $this, 'append_form_script_for_content' ) );
        add_filter( 'block_categories_all', array( $this, 'add_block_categories' ));
    }

    /**
     * スクリプトの読み込み
     *
     * @return void
     */
    function add_scripts() {
        // script --------------------
        wp_enqueue_script(
            'agfs_coypto_cdnjs',
            AGFS_CRYPTO_PATH,
            array(),
            AGFS_VERSION,
            false
        );
        wp_enqueue_script(
            'agfsjs',
            AGFS_SCRIPT_PATH,
            array(),
            AGFS_VERSION,
            false
        );

        wp_register_script( 'agfs_form_submitted_script', false, array(), AGFS_VERSION, true );
        wp_register_script( 'agfs_form_encrypted_script', false, array(), AGFS_VERSION, true );


        // style --------------------
        if(empty(AGFS_CSS_OFF)) {
            wp_enqueue_style(
                'agfsformcss',
                AGFS_FORM_CSS_PATH,
                array(),
                AGFS_VERSION,
                false
            );

            wp_register_style( 'agfsformcssvariable', false, array(), AGFS_VERSION, false);
            wp_enqueue_style( 'agfsformcssvariable' );
            wp_add_inline_style('agfsformcssvariable', $this->get_variable_style());
        };

        wp_enqueue_style(
            'agfsutilitycss',
            AGFS_UTILITY_CSS_PATH,
            array(),
            AGFS_VERSION,
            false
        );
    }

    /**
     * 管理画面で使用するスクリプトの読み込み
     *
    * @return void
     */
    function add_admin_scripts($hook_suffix) {
         // 実装したいページで var_dump( $hook_suffix ) 等して調べる.
        if ( 'settings_page_agfs_settings' === $hook_suffix ) {
            wp_enqueue_script( 'wp-color-picker' );
            wp_enqueue_style( 'wp-color-picker' );

            wp_register_script( 'agfs_settings_script', false, array(), AGFS_VERSION, true );
            wp_enqueue_script( 'agfs_settings_script' );
            wp_add_inline_script('agfs_settings_script', $this->add_setting_script());
        }
    }

    /**
     * ブロックエディタにオリジナルのカテゴリを追加
     *
     * @return array
     */
    function add_block_categories( $categories ) {
        // デフォルトのカテゴリの3番目に差し込み
        array_splice( $categories, 2, 0, array(
            array(
                'slug' => 'agfs-blocks',
                'title' => 'AGF Blocks'
            )
        ));
        return $categories;
    }

    /**
     * 変数上書き用スタイルの取得
     *
     * @return string
     */
    function get_variable_style() {
        $primary_color = esc_html(AGFS_PRIMARY_COLOR);
        $btn_text_color = esc_html(AGFS_BTN_TEXT_COLOR);
        $font_color = esc_html(AGFS_FRONT_COLOR);

        return "body {
            --agfs-primary-color: $primary_color;
            --agfs-btn-text-color: $btn_text_color;
            --agfs-font-color: $font_color;
        }";
    }

    /**
     * オリジナルメニューの追加
     *  @see https://jajaaan.co.jp/wordpress/wordpress-admin-page/
     *
     * @return void
     */
    function add_setting_menu_page() {
        // 親ページを追加
        add_options_page(
            "AGF設定",
            "AGF設定",
            "manage_options",
            "agfs_settings",
            function() {
                require_once AGFS_PLUGIN_PATH.'/view/setting_menu_page.php';
            },
        );

        add_action('admin_init', array($this, 'register_agfs_custom_setting'));
    }

    /**
     * agf設定ページのオプション追加
     *
     * @return void
     */
    function register_agfs_custom_setting() {
        foreach($this->option_setting as $value) {
            register_setting('agfs-custom-setting-group', $value);
        }
    }

    /**
     * 記事コンテンツを加工
     * @param string content - 記事コンテンツ（html）
     * @return string
     */
    function append_form_script_for_content($content) {
        $password = AGFS_DECRYPT_KEY; // ユーザー定義

        if(empty($content)) return $content;

        $dom = new Document();

        $dom->loadHtml($content);
        $forms = $dom->find("form.js-agf-form");

        if(count($forms) <= 0) return $content;

        foreach($forms as $form) {
            $action_url = $form->attr('action');
            $prefix = $form->attr('prefix');
            $thanks_page_url = $form->attr('thankspageurl');

            // 埋め込みscriptプログラムの取得
            $crypt = $this->encrypt($action_url, $password);
            $init_submitted_script = $this->get_init_submitted_script($prefix);
            $encrypted_script = $this->get_encrypted_script($crypt, $password, $prefix, $thanks_page_url);
            $iframe = $this->get_iframe($prefix);

            $this->submitted_script = (empty($init_submitted_script)) ? $this->submitted_script : $init_submitted_script;
            $this->encrypted_script = (empty($encrypted_script)) ? $this->encrypted_script : $encrypted_script;

            add_action('wp_enqueue_scripts', array($this->add_agfs_form_submitted_script($init_submitted_script)), 500);
            add_action('wp_enqueue_scripts', array($this->add_agfs_form_encrypted_script($encrypted_script)), 999);

            add_action('wp_footer', function() use ($iframe) {
                echo wp_kses(
                    $iframe,
                    array(
                        'iframe' => array(
                            'name' => array(),
                            'id' => array(),
                            'onload' => array(),
                            'class' => array()
                        )
                    )
                );
            }, 9999);
        };

        $html_replaces['/<form(.*)action="(.*?)"/'] = '<form${1}action=""';
        $html_replaces['/<form(.*)thankspageurl="(.*?)"/'] = '<form${1}thankspageurl=""';
        return preg_replace(array_keys( $html_replaces ), array_values( $html_replaces ), $content);
    }

    /**
     * 暗号化処理
     *
     * @param string plain - 暗号化したい文字列
     * @param string passphrase - パスワード
     * @return array 暗号化した情報
     */
    function encrypt($plain, $passphrase) {
        $salt = openssl_random_pseudo_bytes(128);
        $iv = openssl_random_pseudo_bytes(16);
        $iterations = 100;
        $key = hash_pbkdf2("sha512", $passphrase, $salt, $iterations, 128);
        $cipher = openssl_encrypt($plain, "AES-256-CBC", hex2bin($key), OPENSSL_RAW_DATA, $iv);
        $data = [
            "cipher" => base64_encode($cipher),
            "salt" => bin2hex($salt),
            "iv" => bin2hex($iv),
        ];
        return $data;
    }

    /**
     * 送信処理用の暗号化されたscriptコードを取得する
     *
     * @return string
     */
    function get_encrypted_script($crypt, $pass, $prefix, $thanks_page_url) {
        $cipher = esc_html($crypt["cipher"]);
        $salt = esc_html($crypt["salt"]);
        $iv = esc_html($crypt["iv"]);
        $event_name = esc_html($prefix)."FormSubmit";
        $submitted_variable = esc_html($prefix)."submitted";
        $thanks_page_url = esc_url($thanks_page_url);

        $jsCode = "
            window.addEventListener('$event_name', (events) => {
                function decrypt(a, b, s, c){
                    const k = CryptoJS.PBKDF2(b, CryptoJS.enc.Hex.parse(s), { hasher: CryptoJS.algo.SHA512, iterations: 100 , keySize: 8});
                    return CryptoJS.AES.decrypt(a, k, { iv: CryptoJS.enc.Hex.parse(c)}).toString(CryptoJS.enc.Utf8);
                };
                const decryptMsg = decrypt(
                    '$cipher',
                    '$pass',
                    '$salt',
                    '$iv'
                );
                const actionUrl = decryptMsg;
                events.detail.setAttribute('action', actionUrl);
                events.detail.submit();
                $submitted_variable = true;
            });";

            $jsCode2 = " function a".esc_html($prefix)."submitFrom(submitted_variable) {
                if(submitted_variable){
                    window.location='$thanks_page_url';
                };
            };";

        $packer = new Packer($jsCode, 'Normal', false, false, true);
        $packed_js = $packer->pack();
        return $packed_js.';'.$jsCode2;
    }


    /**
     * 送信フラッグ変数を初期化するscriptを取得する
     *
     * @return string
     */
    function get_init_submitted_script($prefix) {
        return "let ".esc_html($prefix)."submitted = false;";
    }


    /**
     * 送信完了後のscriptを実行するためのiframeのhtml文字列を生成する
     * @return string
     */
    function get_iframe($prefix) {
        return '<iframe name="'.esc_html($prefix).'_iframe" id="'.esc_html($prefix).'_iframe" class="agf-iframe" onload="a'.esc_html($prefix).'submitFrom('.esc_html($prefix).'submitted)"></iframe>';
    }

    function add_agfs_form_submitted_script($script) {
        wp_enqueue_script( 'agfs_form_submitted_script' );
        wp_add_inline_script('agfs_form_submitted_script', $script);
    }

    function add_agfs_form_encrypted_script($script) {
        wp_enqueue_script( 'agfs_form_encrypted_script' );
        wp_add_inline_script('agfs_form_encrypted_script', $script);
    }

    public function add_setting_script() {
        return "(function() {
                    jQuery(document).ready(function($) {
                        $('.color-picker-hex').wpColorPicker();
            
                        $('#testBtn').click(function() {
                            const testInput = $('#testInput').val();
                            $.ajax({
                                url: '/wp-json/agfs/v1/googleFormSetting',
                                type: 'POST',
                                dataType: 'json',
                                data : {
                                    url:testInput
                                }
                            })
                            .done( (data) => {
                                console.log(data);
                            })
                            .fail( (jqXHR, textStatus, errorThrown) => {
                                console.error('Ajax通信に失敗しました。');
                                console.log('jqXHR          : ' + jqXHR.status); // HTTPステータスを表示
                                console.log('textStatus     : ' + textStatus);    // タイムアウト、パースエラーなどのエラー情報を表示
                                console.log('errorThrown    : ' + errorThrown.message); // 例外情報を表示
                                console.log(textStatus);
                            })
                        });
            
                        $('#decrypt_key_generate_btn').click(function(event) {
                            event.preventDefault();
                            $('#decrypt_key').val(getPass());
                        });
            
                        function getPass(n = 8) {
                            const s ='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
                            return Array.from(Array(n)).map(()=>s[Math.floor(Math.random()*s.length)]).join('');
                        }
                    });
            })()";
    }
}